/*
 Copyright 2013 Adobe Systems Incorporated.  All rights reserved.

Purpose-
This file has the implementation of Image Editing functionality in Live Edit view
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global $, DwImageHud, AuxiliaryExtensionObject, liveViewObject, dwObject, liveExtensionsConfigPath, dwExtensionController, DW_EXTENSION_EVENT, DWLE_DIMENSIONS */

var CONSTANTS = {
    ExtensionID: "dwImageHud",
    DwDivTagName: "dw-div",
    ContainerDiv: "dw-div",
    EscapeKeyCode: 27,
    EnterKeyCode: 13,
    RightArrowKeyCode: 39,
    LeftArrowKeyCode: 37,
    OverlayBorderWidth: 1,
    hudEntryID: 'dwLeAuxHudEntry',
    ExtensionPath: '/DwImageHud',
    Assets: '/assets',
    Folder: '/folder.png',
    TabKeyCode: 9,
    SpaceKeyCode: 32,
    none: "none",
    undef: "undefined",
    HudImageHud: "Image Hud",
    DWUniqueId: "data_liveedit_tagid",
    DWTempId: "data_temp_dwid",
    FILTER: "*.png|*.gif|*.jpg|*.jpeg|*.psd"
};

var DWLE_IMG_IDS = {
    src: "dwLeImgSrc",
    imgSrc: "dwLeImgSrcBtn",
    link: "dwLeImgLink",
    imgLink: "dwLeImgLinkBtn",
    title: "dwLeImgTitle",
    alt: "dwLeImgAlt",
    target: "dwLeImgTarget",
    width: "dwLeImgWidth",
    height: "dwLeImgHeight",
    hudEntryID: "dwLeImgHudEntry"
};

var DWLE_IMG_CLASS = {
    Seperator: " ",
    ImgHudEntry: "dw_imgPropHud",
    ImgSelection: "dw_imgSelection",
    ImgInput: "dw_liveedit_HTMLPI_inputClass",
    ImgInputWandH: "dw_liveedit_HTMLPI_input_Width_Height",
    ImgLabel: "dw_liveedit_HTMLPI_labelClass",
    ImgLabelHeight: "dw_liveedit_HTMLPI_labelForHeight",
    ImgHudRow: "dw_liveedit_HTMLPI_row",
    ImgHudSubRow: "dw_liveedit_HTMLPI_subrow",
    ImgBtnClass: "dw_liveedit_HTMLPI_inputbtnClass",
    ImgContainer: "dw_liveedit_HTMLPIWrapper",
    ImgContainerRight: "dw_liveedit_HTMLPIWrapperRight",
    ImgContainerLeft: "dw_liveedit_HTMLPIWrapperLeft",
    ImgContainerTop: "dw_liveedit_HTMLPIWrapperTop",
    ImgContainerBottom: "dw_liveedit_HTMLPIWrapperBottom",
    NonEditable: "dw_liveedit_HTMLPINonEditable",
    ImgDropDownNone: "dw_liveedit_HTMLPI_donotShow",
    ImgDropDown: "dw_liveedit_HTMLPI_dropdownClass",
    OptionNoneFont: "dw_liveedit_HTMLPI_font",
    Unselectable: "dw_liveedit_HTMLPI_unselectable"
};

var DWLE_IMAGEHUD_HEADLIGHTS = {
    ELV_IH: "ELV IH",
    DW_SHORTCUT: "DWShortcuts",
    OTH_ELV: "OTH ELV",
    SRC: "IH Src",
    TITLE: "IH Title",
    ALT: "IH Alt",
    LINK: "IH Link",
    TARGET: "IH Target",
    WIDTH: "IH Width",
    HEIGHT: "IH Height",
    LAUNCH: "IH Launch",
    BROWSE_SRC: "IH BrowseFile Src",
    BROWSE_LINK: "IH BrowseFile Link",
    CLOSE_SHORTCUT: "IH Closed via Shortcut",
    CLOSE_CLICK: "IH Closed via Click"
};

DwImageHud.prototype = new AuxiliaryExtensionObject();
DwImageHud.prototype.constructor = DwImageHud;
DwImageHud.prototype.baseClass = AuxiliaryExtensionObject.prototype.constructor;

function DwImageHud() {
    'use strict';
    
    //initialize variables

    this.callback_counter = 0; // this is to assign ID's for newly created nodes / modified nodes
    this.callback_map = {}; // map holds the mapping between temporary id's and nodes
    
    this.m_hudName = CONSTANTS.HudImageHud;
    dwExtensionController.addListener(DW_EXTENSION_EVENT.ELEMENT_DIMENSION_CHANGED, this.repositionHUD, this);
    // before we disable the extensions we should be committing the HUDS
	dwExtensionController.addListener(parent.DW_EVENTS.DisableExtensions, this.commit, this);
}

DwImageHud.prototype.getHudTag = function () {
    'use strict';
    return "img";
};

DwImageHud.prototype.fileSave = function () {
    'use strict';
    if (this.getVisibility()) {
        this.fileSavePending = 1;
        this.commit();
    }
};

DwImageHud.prototype.fileSaveAs = function () {
    'use strict';
    if (this.getVisibility()) {
        this.fileSavePending = 0;
        this.commit();
    }
};

/**
 * @private createLabel
 * Create the label
 * @Param toDiv     :to which this created label should be made as child 
 * @Param labelText :actual label value
 * @Param labelClass:css for label
 * @Param forInputId:inputID corresponding to the label
 */
DwImageHud.prototype.createLabel = function (toDiv, labelText, labelClass, forInputId, tooltip) {
    'use strict';
    var labelElement = document.createElement("label");
    labelElement.innerHTML = labelText;
    labelElement.title = tooltip || "";
    labelElement.htmlFor = forInputId;
    labelElement.setAttribute("class", DWLE_IMG_CLASS.Reset + " " + labelClass);
    toDiv.appendChild(labelElement);
};
/**
 * @private createInput
 * Create input component with given values 
 * @Param toDiv     : to which this created label should be made as child
 * @Param ofType    : file/text/image
 * @Param idOfInput : id to be assigned
 * @Param inputClass: class to be applied
 * @Param placeholderText: placeholderText
 * @Param readOnlyFlag : whether the element is readOnly or writable
 * @Return an InputElement created with given arguments as values
 * @Note DWLE_IMG_CLASS.Reset 
 *         Needs to be applied on all image HUD elements 
 *          In order to make sure that we are not hit by globally applied styles
 */
DwImageHud.prototype.createInput = function (toDiv, ofType, idOfInput, inputClass, placeHolderText, readOnlyFlag, attr, isParent) {
    'use strict';
    var inputElement = document.createElement("input");
    inputElement.id = idOfInput;
    inputElement.placeholder = placeHolderText;
    inputElement.setAttribute("type", ofType);
    inputElement.setAttribute("class", inputClass);
    if (readOnlyFlag === true) {
        inputElement.setAttribute('readonly', 'true');
    }
    //this is done to make sure that drag is differentiated from text selection inside input box
    inputElement.onmousedown = function (e) {
        e.stopPropagation();
    };
    toDiv.appendChild(inputElement);
    
    var self = this;
    if (ofType === "text") {
        inputElement.onfocus = function () {
            if (attr) {
                var element = self.m_currentSelectedElement;
                if (element) {
                    if (isParent) {
                        element = element.parent;
                    }
                    if (element && (element.getAttribute(attr) || inputElement.value) && (element.getAttribute(attr) !== inputElement.value)) {
                        self.commit();
                    }
                }
            }
        };
    }
    return inputElement;
};
/**
 * @private createDropDown
 * create DropDown with given values
 * @Param toDiv     : to which this created label should be made as child
 * @Param idApplied : id to be assigned
 * @Param classToBeApplied: class to be applied
 * @Param placeholderText: placeholderText
 * @Param values : values to be populated in dropdown
 * @Return an Dropdown Element created with given arguments as values
 */
DwImageHud.prototype.createDropDown = function (toDiv, idApplied, classToBeApplied, values) {
    'use strict';
    var selectElement = document.createElement("select");
    selectElement.id = idApplied;
    selectElement.setAttribute("class", classToBeApplied);
    selectElement.setAttribute("disabled", "disabled");
    var index, opt;
    var option = document.createElement("option");
    option.innerHTML = "none";
    option.setAttribute("class", DWLE_IMG_CLASS.OptionNoneFont);
    option.setAttribute("selected", "selected");
    selectElement.appendChild(option);

    for (index = 0; index < values.length; ++index) {
        opt = document.createElement("option");
        opt.value = values[index];
        opt.text = values[index];
        selectElement.appendChild(opt);
    }
    toDiv.appendChild(selectElement);
    return selectElement;
};

/**
 * @private createContainer
 * create a DW-DIV with given class applied
 * @Param classToBeApplied:class to be applied on dw-div
 * @Return return dw-div with the class applied
 */
DwImageHud.prototype.createContainer = function (classToBeApplied) {
    'use strict';
    var newContainerDiv = document.createElement(CONSTANTS.ContainerDiv);
    newContainerDiv.setAttribute("class", classToBeApplied);
    return newContainerDiv;
};

/**
 * @private updateHudFields
 * Update HUD Fields with respect to the current element under edit
 * @Param elem : Current Element Under Edit.
*/
DwImageHud.prototype.updateHudFields = function (elem) {
    'use strict';
    if (!elem) {
        return;
    }
    if (elem.src) {
        var sourceElement = document.getElementById(DWLE_IMG_IDS.src);
        sourceElement.value = elem.getAttribute("src");
        sourceElement.title = elem.getAttribute("src");
    }
    if (elem.alt) {
        var altElement = document.getElementById(DWLE_IMG_IDS.alt);
        altElement.value = elem.alt;
        altElement.title = elem.alt;
    }
    if (elem.title) {
        var titleElement = document.getElementById(DWLE_IMG_IDS.title);
        titleElement.value = elem.title;
        titleElement.title = elem.title;
    }
    // for Link tag and target
    var parent = elem.parentNode;
    var targetElement = document.getElementById(DWLE_IMG_IDS.target);
    if (parent && parent.tagName === "A") {
        var href = parent.getAttribute("href");
        if (href !== null) {

            var linkElement = document.getElementById(DWLE_IMG_IDS.link);
            linkElement.value = href;
            linkElement.title = href;
            var target = parent.getAttribute("target");

            if (this.getEditable() === true) {
                targetElement.removeAttribute("disabled");
                targetElement.setAttribute("class", DWLE_IMG_CLASS.ImgDropDown);
            }
            if (target !== null) {

                targetElement.value = target;
            } else {

                targetElement.value = "none";
            }
        } else {
            targetElement.setAttribute("disabled", "disabled");
            targetElement.setAttribute("class", DWLE_IMG_CLASS.ImgDropDown + " " + DWLE_IMG_CLASS.ImgDropDownNone);
            targetElement.value = "none";
        }
    } else {
        targetElement.value = "none";
    }
    var heightOfElement = elem.getAttribute("height");
    if (heightOfElement) {
        var heightElement = document.getElementById(DWLE_IMG_IDS.height);
        heightElement.value = heightOfElement;
        heightElement.title = heightOfElement;
    }
    var widhtOfElement = elem.getAttribute("width");
    if (widhtOfElement) {
        var widthElement = document.getElementById(DWLE_IMG_IDS.width);
        widthElement.value = widhtOfElement;
        widthElement.title = widhtOfElement;
    }
};

/**
* @private handleArrowKeys
* handles Right & Left Arrow keys for input elements
* If left arrow key is pressed and cursor is at the end of text, or
* Right arrow key is pressed and cursor is at the beginning of text,
* then nothing should occur
* @Param e:keypress event
* @Param elem:the HTML element on which key is pressed
*/
DwImageHud.prototype.handleArrowKeys = function (e, elem) {
    'use strict';
    if (e.keyCode === CONSTANTS.RightArrowKeyCode) {
        var elemLen = elem.value.length;
        //check if cursor is at the end of elem text
        if (elem.selectionStart === elemLen) {
            e.stopPropagation();
            e.preventDefault();
        }
    } else if (e.keyCode === CONSTANTS.LeftArrowKeyCode) {
        //check if cursor is at the start of src text
        if (elem.selectionEnd === 0) {
            e.stopPropagation();
            e.preventDefault();
        }
    }
};

/**
* @private handleArrowKeysForButtons
* handles Right & Left Arrow keys for buttons
* If left or right arrow key is pressed, nothing should occur
* @Param e:keypress event
*/
DwImageHud.prototype.handleArrowKeysForButtons = function (e) {
    'use strict';
    if (e.keyCode === CONSTANTS.RightArrowKeyCode || e.keyCode === CONSTANTS.LeftArrowKeyCode) {
        e.stopPropagation();
        e.preventDefault();
    }
};


/**
 * @private populateContent
 * populate content inside the Image HUD container div.
 * this is responsible for both editable and non editable cases
 * creation of each of the component and also attaching event listeners and 
 * handling values should be done from this function
 */
DwImageHud.prototype.populateContent = function () {
    'use strict';
    var container = this.getHudContainer();
    container.innerHTML = "";

    /* create a container div to holds the content*/
    this.m_hudContentDiv = this.createContainer(DWLE_IMG_CLASS.ImgContainer + " " + DWLE_IMG_CLASS.ImgContainerRight);
    AuxiliaryExtensionObject.prototype.EnableDragOnElement.call(this, this.m_hudContentDiv);

    //Create containers for each of the rows
    var imgSrcDiv = this.createContainer(DWLE_IMG_CLASS.ImgHudRow);
    var imgLinkDiv = this.createContainer(DWLE_IMG_CLASS.ImgHudRow);
    var imgTitleDiv = this.createContainer(DWLE_IMG_CLASS.ImgHudRow);
    var imgAltDiv = this.createContainer(DWLE_IMG_CLASS.ImgHudRow);
    var imgTargetDiv = this.createContainer(DWLE_IMG_CLASS.ImgHudRow);
    var imgWidthDiv = this.createContainer(DWLE_IMG_CLASS.ImgHudSubRow);
    var imgHeightDiv = this.createContainer(DWLE_IMG_CLASS.ImgHudSubRow);
    var containerForWandH = this.createContainer(DWLE_IMG_CLASS.ImgHudRow);

    var classToApplyOnInput = DWLE_IMG_CLASS.ImgInput;
    var classToApplyOnInputForWH = DWLE_IMG_CLASS.ImgInputWandH;
    var readOnly = false;
    if (this.getEditable() === false) {
        classToApplyOnInput += " " + DWLE_IMG_CLASS.NonEditable;
        classToApplyOnInputForWH += " " + DWLE_IMG_CLASS.NonEditable;
        readOnly = true;
    }

    //Image Source Div
    this.createLabel(imgSrcDiv, this.locStrings.srcLabel, DWLE_IMG_CLASS.ImgLabel, DWLE_IMG_IDS.src, this.locStrings.srcTooltip);
    var inputElemSrc = this.createInput(imgSrcDiv, "text", DWLE_IMG_IDS.src, classToApplyOnInput, "", readOnly);

    if (this.getEditable() === true) {
        // this is to achieve shift+tab should go back to last filed and should not navigate in the page.
        // it must be onkeydown as tab does not get captured using onkeyup
        inputElemSrc.onkeydown = function (e) {

            if (e.keyCode === CONSTANTS.TabKeyCode) {
                if (e.shiftKey) {
                    //Focus previous input
                    this.focusHudField(DWLE_IMG_IDS.height);
                    e.preventDefault();
                }
                this.updateSrc(true);
                e.stopPropagation();
            } else if (e.keyCode === CONSTANTS.EnterKeyCode) {
                this.updateSrc(true);
                e.stopPropagation();
            } else {
                this.handleArrowKeys(e, inputElemSrc);
            }
        }.bind(this);

        var inputElemSrcBtn = this.createInput(imgSrcDiv, "image", DWLE_IMG_IDS.imgSrc, DWLE_IMG_CLASS.ImgBtnClass, "", readOnly, "src");
        inputElemSrcBtn.title = this.locStrings.browseTooltip;
        inputElemSrcBtn.setAttribute("src", liveExtensionsConfigPath + CONSTANTS.ExtensionPath + CONSTANTS.Assets + CONSTANTS.Folder);

        inputElemSrcBtn.onclick = function () {
            inputElemSrcBtn.blur();
            // this needs to be done because this element has focus after showing browse dialog and gets all key events(including enter)
            // and infinitely show up browse dialog
            this.launchFileSelection(this.locStrings.srcLabel);
        }.bind(this);
        
        
        var invokeSelectSoureDialogOnKeyUp = function (e) {
            
            if (e.keyCode === CONSTANTS.SpaceKeyCode) {
                if (this.getVisibility()) {
                    inputElemSrcBtn.blur();
                    // this needs to be done because this element has focus after showing browse dialog and gets all key events(including enter)
                    // and infinitely show up browse dialog
                    this.launchFileSelection(this.locStrings.srcLabel);
                    e.preventDefault();
                    e.stopPropagation();
                }
            }
        };
        
        
        // This is for Mac
        var invokeSelectSourceDialog = function (e) {
            if (this.getVisibility()) {
                inputElemSrcBtn.blur();
                // this needs to be done because this element has focus after showing browse dialog and gets all key events(including enter)
                // and infinitely show up browse dialog
                this.launchFileSelection(this.locStrings.srcLabel);
                e.preventDefault();
                e.stopPropagation();
            }
        };


        inputElemSrcBtn.onkeydown = function (e) {
            if (e.keyCode === CONSTANTS.EnterKeyCode) {
                invokeSelectSourceDialog(e);
            }
        }.bind(this);

        inputElemSrcBtn.onkeydown = function (e) {
            this.handleArrowKeysForButtons(e);
        }.bind(this);

        inputElemSrcBtn.addEventListener("keyup", invokeSelectSoureDialogOnKeyUp.bind(this), true);// enter/space bar should bring the file selection dialog
        
    }
    // Image Link Div
    this.createLabel(imgLinkDiv, this.locStrings.linkLabel, DWLE_IMG_CLASS.ImgLabel, DWLE_IMG_IDS.link, this.locStrings.linkTooltip);
    var inputElemLink = this.createInput(imgLinkDiv, "text", DWLE_IMG_IDS.link, classToApplyOnInput, CONSTANTS.undef, readOnly, "href", true);
    if (this.getEditable() === true) {
        var inputElemLinkBtn = this.createInput(imgLinkDiv, "image", DWLE_IMG_IDS.imgLink, DWLE_IMG_CLASS.ImgBtnClass, "", readOnly);
        inputElemLinkBtn.title = this.locStrings.browseTooltip;
        inputElemLink.onkeydown = function (e) {
            if (e.keyCode === CONSTANTS.TabKeyCode) {
                this.updateLink("");
                e.stopPropagation();
            } else {
                this.handleArrowKeys(e, inputElemLink);
            }
        }.bind(this);
        inputElemLink.onkeyup = function (e) {
            if (e.keyCode === CONSTANTS.EnterKeyCode) {
                this.updateLink("");
                e.stopPropagation();
            } else {
                this.updateTargetDropdown();
            }
        }.bind(this);

        inputElemLinkBtn.setAttribute("src", liveExtensionsConfigPath + CONSTANTS.ExtensionPath + CONSTANTS.Assets + CONSTANTS.Folder);
        inputElemLinkBtn.onclick = function () {
            inputElemLinkBtn.blur(); //same reason as link
            this.launchFileSelection(this.locStrings.linkLabel);
        }.bind(this);
        
        var invokeSelectLinkDialogOnKeyUp = function (e) {
            
            if (e.keyCode === CONSTANTS.SpaceKeyCode) {
                
                if (this.getVisibility()) {
                    inputElemLinkBtn.blur(); //same reason as link
                    this.launchFileSelection(this.locStrings.linkLabel);
                    e.preventDefault();
                    e.stopPropagation();
                }
            }
        };
        
        var invokeSelectLinkDialog = function (e) {
            if (this.getVisibility()) {
                inputElemLinkBtn.blur(); //same reason as link
                this.launchFileSelection(this.locStrings.linkLabel);
                e.preventDefault();
                e.stopPropagation();
            }
        };
        // this is to make sure that tabbing is insiode the hud
        inputElemLinkBtn.onkeydown = function (e) {
            if (e.keyCode === CONSTANTS.EnterKeyCode) {
                invokeSelectLinkDialog(e);
            }
        }.bind(this);
        inputElemLinkBtn.onkeydown = function (e) {
            this.handleArrowKeysForButtons(e);
        }.bind(this);
        
        inputElemLinkBtn.addEventListener("keyup", invokeSelectLinkDialogOnKeyUp.bind(this), true);// enter/space bar should bring the file selection dialog
    }
    //Image Title Div
    this.createLabel(imgTitleDiv, this.locStrings.titleLabel, DWLE_IMG_CLASS.ImgLabel, DWLE_IMG_IDS.title, this.locStrings.titleTooltip);
    var inputElemTitle = this.createInput(imgTitleDiv, "text", DWLE_IMG_IDS.title, classToApplyOnInput, CONSTANTS.none, readOnly, "title");

    // img Alt Div
    this.createLabel(imgAltDiv, this.locStrings.altLabel, DWLE_IMG_CLASS.ImgLabel, DWLE_IMG_IDS.alt, this.locStrings.altTooltip);
    var inputElemAlt = this.createInput(imgAltDiv, "text", DWLE_IMG_IDS.alt, classToApplyOnInput, CONSTANTS.none, readOnly, "alt");


    // image target div
    this.createLabel(imgTargetDiv, this.locStrings.targetLabel, DWLE_IMG_CLASS.ImgLabel, DWLE_IMG_IDS.target, this.locStrings.targetTooltip);
    var inputElemTarget = this.createDropDown(imgTargetDiv, DWLE_IMG_IDS.target, DWLE_IMG_CLASS.ImgDropDown + " " + DWLE_IMG_CLASS.ImgDropDownNone, ["_blank", "_self", "_parent", "_top"]);

    // image width div
    this.createLabel(imgWidthDiv, this.locStrings.widthLabel, DWLE_IMG_CLASS.ImgLabel, DWLE_IMG_IDS.width, this.locStrings.widthTooltip);
    var inputElemWidth = this.createInput(imgWidthDiv, "text", DWLE_IMG_IDS.width, classToApplyOnInputForWH, CONSTANTS.none, readOnly, "width");
    // image height div
    this.createLabel(imgHeightDiv, this.locStrings.heightLabel, DWLE_IMG_CLASS.ImgLabelHeight, DWLE_IMG_IDS.height, this.locStrings.HeightTooltip);
    var inputElemHeight = this.createInput(imgHeightDiv, "text", DWLE_IMG_IDS.height, classToApplyOnInputForWH, CONSTANTS.none, readOnly, "height");

    // container for both width and height
    containerForWandH.appendChild(imgWidthDiv);
    containerForWandH.appendChild(imgHeightDiv);

    if (this.getEditable() === true) {
        inputElemTitle.onkeydown = function (e) {
            if (e.keyCode === CONSTANTS.EnterKeyCode || e.keyCode === CONSTANTS.TabKeyCode) {
                this.updateTitle();
                e.stopPropagation();
            } else {
                this.handleArrowKeys(e, inputElemTitle);
            }
        }.bind(this);

        inputElemWidth.onkeypress = function (e) {
            // user must not be able to enter non numeric characters in this fields.
            this.validateInputText(e);
        }.bind(this);


        inputElemWidth.onkeydown = function (e) {
            if (e.keyCode === CONSTANTS.EnterKeyCode || e.keyCode === CONSTANTS.TabKeyCode) {
                this.updateWidth(true);
                e.stopPropagation();
            } else {
                this.handleArrowKeys(e, inputElemWidth);
            }
        }.bind(this);

        inputElemHeight.onkeypress = function (e) {
            // user must not be able to enter non numeric characters in this fields.
            this.validateInputText(e);

        }.bind(this);

        inputElemHeight.onkeydown = function (e) {

            if (e.keyCode === CONSTANTS.EnterKeyCode) {
                this.updateHeight(true);
                e.stopPropagation();
            } else if (e.keyCode === CONSTANTS.TabKeyCode) {
                this.updateHeight(true);
                if (!e.shiftKey) {
                    this.focusHudField(DWLE_IMG_IDS.src);
                } else {
                    this.focusHudField(DWLE_IMG_IDS.width);
                }
                e.stopPropagation();
                e.preventDefault();
            } else {
                this.handleArrowKeys(e, inputElemHeight);
            }

        }.bind(this);

        inputElemAlt.onkeydown = function (e) {
            if (e.keyCode === CONSTANTS.EnterKeyCode || e.keyCode === CONSTANTS.TabKeyCode) {
                this.updateAlt();
                e.stopPropagation();
            } else {
                this.handleArrowKeys(e, inputElemAlt);
            }
        }.bind(this);

        inputElemTarget.onkeydown = function (e) {
            if (e.keyCode === CONSTANTS.TabKeyCode) {
                this.updateLink("");
                e.stopPropagation();
            }
        }.bind(this);

        inputElemTarget.onchange = function () {
            // On target change we need to update link because if link is modified and not committed we will not even have anchor tag in the page.
            this.updateLink("");

        }.bind(this);

        inputElemTarget.onmousedown = function (e) {
            e.stopPropagation();
        };

    }
    this.m_hudContentDiv.appendChild(imgSrcDiv);
    this.m_hudContentDiv.appendChild(imgTitleDiv);
    this.m_hudContentDiv.appendChild(imgAltDiv);
    this.m_hudContentDiv.appendChild(imgLinkDiv);
    this.m_hudContentDiv.appendChild(imgTargetDiv);
    this.m_hudContentDiv.appendChild(containerForWandH);
    // Add image hud related css 
    var style = document.createElement('style');
    style.type = 'text/css';
    style.innerHTML = window.liveEditImageHudCss;
    container.appendChild(style);
    //add actual hud content
    container.appendChild(this.m_hudContentDiv);

    this.m_initialized = true;
};

/**
 * @private updateTargetDropdown
 * update the target dropdown depending on the value in link. 
 * target does not make sense when we dont have a link associated.
 * @Param evt     : keystroke event that we received.
 */
DwImageHud.prototype.updateTargetDropdown = function () {
    'use strict';
    var value = document.getElementById(DWLE_IMG_IDS.link).value;
    var targetElement = document.getElementById(DWLE_IMG_IDS.target);
    if (value === "" || value === "undefined") {
        targetElement.setAttribute("disabled", "disabled");
        targetElement.setAttribute("class", DWLE_IMG_CLASS.ImgDropDown + " " + DWLE_IMG_CLASS.ImgDropDownNone);
        targetElement.value = "none";
    } else {
        targetElement.removeAttribute("disabled");
        targetElement.setAttribute("class", DWLE_IMG_CLASS.ImgDropDown);
    }
};

DwImageHud.prototype.launchFileSelection = function (type) {
    'use strict';
    var argObj = {};
    argObj.filter = "";
    if (type === this.locStrings.srcLabel) {
        argObj.operation = this.locStrings.fileDialogType;
        argObj.subOp = this.locStrings.fileTitleSrc;
        argObj.filter = CONSTANTS.FILTER;
        argObj.callback = function (value) {
            var srcElement = document.getElementById(DWLE_IMG_IDS.src);
            if (value && srcElement.value !== value) {
                srcElement.value = value;
                srcElement.title = value;
                this.updateSrc(true);
            }
            this.stopHide = false;
            this.focusHudField(DWLE_IMG_IDS.imgSrc);
        }.bind(this);
        dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.ELV_IH, DWLE_IMAGEHUD_HEADLIGHTS.BROWSE_SRC);
    } else if (type === this.locStrings.linkLabel) {
        argObj.operation = this.locStrings.fileDialogType;
        argObj.subOp = this.locStrings.fileTitleLink;
        argObj.callback = function (value) {
            this.updateLink(value);
            this.focusHudField(DWLE_IMG_IDS.imgLink);
            this.stopHide = false;
        }.bind(this);
        dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.ELV_IH, DWLE_IMAGEHUD_HEADLIGHTS.BROWSE_LINK);
    } else {
        return;
    }
    this.stopHide = true;
    dwObject.dwBrowseForFileURL(argObj.operation, argObj.subOp, argObj.callback, argObj.filter, false);
};

/*
function : focusHudField
Arguments: current Image Element

give focus to hud field

*/
DwImageHud.prototype.focusHudField = function (id) {
    'use strict';

    var focusElement = document.getElementById(id);
    if (focusElement) {
        focusElement.focus();
    }
};

DwImageHud.prototype.toggleAuxHud = function (message) {
    'use strict';
    AuxiliaryExtensionObject.prototype.toggleAuxHud.call(this, message);
    var elem = this.getCurrentElement();
    if (this.getVisibility()) {
        var srcElement = document.getElementById(DWLE_IMG_IDS.src);
        if (this.getEditable() === true) {
            srcElement.focus();
        }
        // this is to compute natural height and width of an element after changing.
        elem.addEventListener("load", this.postImageLoad.bind(this), true);
    } else {
        // this is to compute natural height and width of an element after changing.
        elem.removeEventListener("load", this.postImageLoad.bind(this), true);
    }
    if (message.shortcut) {
        if (this.getVisibility()) {
            dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.DW_SHORTCUT, DWLE_IMAGEHUD_HEADLIGHTS.LAUNCH);
        } else {
            dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.OTH_ELV, DWLE_IMAGEHUD_HEADLIGHTS.CLOSE_SHORTCUT);
        }
    } else {
        if (this.getVisibility()) {
            dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.ELV_IH, DWLE_IMAGEHUD_HEADLIGHTS.LAUNCH);
        } else {
            dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.OTH_ELV, DWLE_IMAGEHUD_HEADLIGHTS.CLOSE_CLICK);
        }
    }
};

DwImageHud.prototype.getExtensionId = function () {
    'use strict';
    return CONSTANTS.ExtensionID;
};

/**
 * @override commit
 * Commit the values of each of the attribute in Image Hud. 
 * @Param None
*/
DwImageHud.prototype.commit = function () {
    'use strict';

    if (this.getVisibility() === false) {
        return;
    }
    this.updateSrc(false);
    this.updateTitle();
    this.updateAlt();
    this.updateWidth(false);//false indicates that dont redraw the view
    this.updateHeight(false);
    // Link needs to be done at the end since it leads to creation of new elements. This will update the target as well.
    this.updateLink("");

//	if (this.attributeChanged) {
//		this.m_liveEditEvent.fire({
//			type: DW_LIVEEDIT_EVENTS.ElementChanged
//		});
//		this.attributeChanged = false;
//	}
};

/**
 * @private positionHud
 * Position the HUD according to element's width, height and position
 * @Param elemRect : contains computed width,height,top,left 
 * @Note This positions the HUD with respect to the space available around the Image.
 * DWLE_DIMENSIONS.AuxHudOffSet : 10px  --> for the arrow to come before the HUD
 */
DwImageHud.prototype.positionHud = function (elemRect) {
    'use strict';
    //cache position of hud because offsetwidth and offsetheight gives 0 when element display is made to none
    if (this.m_currentHudHeight === 0 && this.m_currentHudWidth === 0) {
        this.m_currentHudHeight = this.m_hudContentDiv.offsetHeight;
        this.m_currentHudWidth = this.m_hudContentDiv.offsetWidth;
    }
    // by default try to position on right ---> top remains the same as element and width is elementleft + elementwidth
    this.m_hudContentDiv.style.top = (elemRect.top) + 'px';
    this.m_hudContentDiv.style.left = (elemRect.left) + (elemRect.width + DWLE_DIMENSIONS.AuxHudOffSet) + 'px';

    // right Side Check contains max horizontal position needed to place the hud on right side
    var rightSideCheck = elemRect.left + elemRect.width + this.m_currentHudWidth - window.parent.pageXOffset;
    // height Check contains max vertical position takes to place the hud at same level
    var heightCheck = elemRect.top - window.parent.scrollY + this.m_currentHudHeight;
    /*
     if our righ side position is greater than current screen horizontal bounds or height is more than current screen vertical bound
     explore other options
     if there is no space on right or element Hud is taking more space than the elmenet width then place it in other position
    */
    if (rightSideCheck > window.parent.innerWidth || heightCheck > window.parent.innerHeight || elemRect.width < this.elementHudWidth) {
        // elementleft -( HUD space) - (page amount already scrolled) should be in current screen bounds
        var leftSideCheck = elemRect.left - window.parent.pageXOffset - this.m_currentHudWidth - DWLE_DIMENSIONS.AuxHudOffSet;
        if (leftSideCheck > 0 && (heightCheck < window.parent.innerHeight)) {
            this.m_hudContentDiv.style.left = leftSideCheck + window.parent.pageXOffset + 'px';
            this.m_hudContentDiv.removeAttribute("class");
            this.m_hudContentDiv.setAttribute("class", DWLE_IMG_CLASS.ImgContainer + " " + DWLE_IMG_CLASS.ImgContainerLeft);
            // this class is to make the arrow appear on right side of the hud
        } else {

            this.m_hudContentDiv.style.left = (elemRect.left) + "px";
            // elementtop - (Space for HUD) - editable selectors height should in curren screen offset
            var topCorner = elemRect.top - this.m_currentHudHeight - DWLE_DIMENSIONS.AuxHudOffSet - this.elementHudHeight;
            if (topCorner - window.parent.scrollY > 0) {
                // top case                
                this.m_hudContentDiv.style.top = topCorner + 'px';
                this.m_hudContentDiv.removeAttribute("class");
                this.m_hudContentDiv.setAttribute("class", DWLE_IMG_CLASS.ImgContainer + " " + DWLE_IMG_CLASS.ImgContainerTop);
            } else {
                var bottom = elemRect.top + elemRect.height + this.m_currentHudHeight + DWLE_DIMENSIONS.AuxHudOffSet - window.parent.scrollY;

                if (bottom < window.parent.innerHeight) { //bottom case
                    var heightToPlace = elemRect.top + elemRect.height + DWLE_DIMENSIONS.AuxHudOffSet;
                    //if the editable selectors is on bottom add that height also
                    if (this.elementHudPosOnTop === false) {
                        heightToPlace += this.elementHudHeight;
                    }
                    this.m_hudContentDiv.style.top = heightToPlace + 'px';
                    this.m_hudContentDiv.removeAttribute("class");
                    this.m_hudContentDiv.setAttribute("class", DWLE_IMG_CLASS.ImgContainer + " " + DWLE_IMG_CLASS.ImgContainerBottom);
                } else {
                    //cant place anywhere --> placing on top of element
                    //if editable selectors is on top 
                    if (this.elementHudPosOnTop === true) {
                        this.m_hudContentDiv.style.top = elemRect.top + DWLE_DIMENSIONS.AuxHudOffSet + 'px';
                        this.m_hudContentDiv.style.left = elemRect.left + 'px';
                        this.m_hudContentDiv.removeAttribute("class");
                        this.m_hudContentDiv.setAttribute("class", DWLE_IMG_CLASS.ImgContainer + " " + DWLE_IMG_CLASS.ImgContainerBottom);
                    } else {
                        //if the editable selectors is on bottom.
                        this.m_hudContentDiv.style.top = elemRect.top + elemRect.height - this.m_currentHudHeight - DWLE_DIMENSIONS.AuxHudOffSet + 'px';
                        this.m_hudContentDiv.style.left = elemRect.left + 'px';
                        this.m_hudContentDiv.removeAttribute("class");
                        this.m_hudContentDiv.setAttribute("class", DWLE_IMG_CLASS.ImgContainer + " " + DWLE_IMG_CLASS.ImgContainerTop);
                    }
                }
            }
        }
    } else {
        this.m_hudContentDiv.removeAttribute("class");
        this.m_hudContentDiv.setAttribute("class", DWLE_IMG_CLASS.ImgContainer + " " + DWLE_IMG_CLASS.ImgContainerRight);
    }
    this.m_hudContentDiv.style.visibility = "visible";
};

DwImageHud.prototype.getHudContainer = function () {
    'use strict';
    return this.m_hudContainer;
};

DwImageHud.prototype.initialize = function () {
    'use strict';
    AuxiliaryExtensionObject.prototype.initialize.call(this);
    
    this.globalController = window.parent.globalController;
    
    this.locStrings = {};

    this.locStrings.browseTooltip = this.globalController.getLocalizedString('Common_Browse_Tooltip');
    this.locStrings.fileDialogType = this.globalController.getLocalizedString('Common_DialogType');
	this.locStrings.fileTitleSrc = this.globalController.getLocalizedString('Common_FileTitleSrc');
	this.locStrings.fileTitleLink = this.globalController.getLocalizedString('Common_FileTitleLink');
    
    this.locStrings.srcLabel = this.globalController.getLocalizedString('Img_Hud_Src_Label');
    this.locStrings.linkLabel = this.globalController.getLocalizedString('Img_Hud_Link_Label');
    this.locStrings.titleLabel = this.globalController.getLocalizedString('Img_Hud_Title_Label');
    this.locStrings.altLabel = this.globalController.getLocalizedString('Img_Hud_Alt_Label');
    this.locStrings.targetLabel = this.globalController.getLocalizedString('Img_Hud_Target_Label');
    this.locStrings.widthLabel = this.globalController.getLocalizedString('Img_Hud_Width_Label');
    this.locStrings.heightLabel = this.globalController.getLocalizedString('Img_Hud_Height_Label');
    
    this.locStrings.srcTooltip = this.globalController.getLocalizedString('Img_Hud_Src_Tooltip');
    this.locStrings.linkTooltip = this.globalController.getLocalizedString('Common_Link_ToolTip');
    this.locStrings.titleTooltip = this.globalController.getLocalizedString('Img_Hud_Title_Tooltip');
    this.locStrings.altTooltip = this.globalController.getLocalizedString('Img_Hud_Alt_Tooltip');
    this.locStrings.targetTooltip = this.globalController.getLocalizedString('Img_Hud_Target_Tooltip');
    this.locStrings.widthTooltip = this.globalController.getLocalizedString('Img_Hud_W_Tooltip');
    this.locStrings.HeightTooltip = this.globalController.getLocalizedString('Img_Hud_H_Tooltip');
    
    this.stopHide = false;
};
/**
 * @private repositionHUD
 * respotion the hud
 * @Param none
 */
DwImageHud.prototype.repositionHUD = function () {
    'use strict';
    var curElem = this.getCurrentElement();
    // do not reposition if the hud has been dragged
	if (!curElem || this.m_hudDragged === true) {
        return;
    }
    var elemRect = liveViewObject.getElementRect(curElem);
    this.positionHud(elemRect);
};
/**
 * @private updateSrc
 * if we pass a string other than "" it will take that value otherwise it will read the value of the filed from hud.
 * @Param srcValue: "" if it is from file dialog and value in the input box if is from HUD Input Boxes
 */
DwImageHud.prototype.updateSrc = function (updateUI) {
    'use strict';

    this.updateElement("src", DWLE_IMG_IDS.src);
    // After the source has changed we need to redraw the whole stuff again
    if (updateUI === true) {
        AuxiliaryExtensionObject.prototype.elementDimensionChanged.call(this);
//        if (this.attributeChanged) {
//			this.m_liveEditEvent.fire({
//				type: DW_LIVEEDIT_EVENTS.ElementChanged
//			});
//			this.attributeChanged = false;
//		}
    }
};

/**
 * @private postImageLoad
 * After the image loading compute the width and height of the image 
 * this has to be done only when there is a width or height attribute present already.
 * Mocking design view behavior
 * @Param none
 */
DwImageHud.prototype.postImageLoad = function () {
    'use strict';
    var activeFocus = document.activeElement;
    var elem = this.getCurrentElement();
    if (!elem) {
        this.logDebugMsg("function begin : ImageHUD : postImageLoad : Failed");
        return;
    }
    var heightElement = document.getElementById(DWLE_IMG_IDS.height);
    var widthElement = document.getElementById(DWLE_IMG_IDS.width);
    if (heightElement.value) {
        heightElement.value = elem.naturalHeight;
    }
    if (widthElement.value) {
        widthElement.value =  elem.naturalWidth;
    }
    this.updateWidth(false);//false indicates that dont redraw the view
    this.updateHeight(true);
    if (activeFocus.type && activeFocus.type === "image") {
        // browse button had focus. restore it.
        activeFocus.focus();
    } else {
        this.focusHudField(DWLE_IMG_IDS.src);
    }
};
/**
 * @private updateLink
 * Wraps the image tag with an anchor tag.
 * @Param linkValue: "" if it is from file dialog and value in the input box if is from HUD Input Boxes
 */
DwImageHud.prototype.updateLink = function (linkValue) {
    'use strict';
    var linkElement = document.getElementById(DWLE_IMG_IDS.link);
    if (linkValue === "") {
        linkValue = linkElement.value;
		linkElement.title = linkValue;
    } else {

        linkElement.value = linkValue;
        linkElement.title = linkValue;
    }
    // if link is deleted or empty --> target should also be empty
    if (linkValue === "" || linkValue === "undefined") {
        document.getElementById(DWLE_IMG_IDS.target).value = "none";
    }
    var elem = this.getCurrentElement();
    var parent = elem.parentNode;
    var target = null,
        linkNode;
    var imageId = elem.getAttribute(CONSTANTS.DWUniqueId);
    var newOuterHTML = "";
    var newCallbackObj = {};
    // if LinkValue is empty --> it is deleted or not there at all
    // else linkValue is non empty --> it is associated newly or existing link is modified.
    if (linkValue !== "") {
        //if link exists previously and we updated it -->get parent DWId and push the changes
        if (parent.tagName === "A") {
            var existingHref = parent.getAttribute("href");
            //if the value is not changed dont do anything
            if (existingHref && existingHref === linkValue) {
                this.updateTarget();
                return;
            }
            parent.setAttribute("href", linkValue);
            dwObject.updateDWDocumentElementAttr(parent, 'href', linkValue, null, imageId);
            dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.ELV_IH, DWLE_IMAGEHUD_HEADLIGHTS.LINK);
            this.updateTarget();
            return;
        }
        // we are associating links newly.
        // add <a> tag in between image and its parent
        linkNode = document.createElement("a");
        linkNode.setAttribute("href", linkValue);
        elem.parentNode.replaceChild(linkNode, elem);
        linkNode.appendChild(elem);
        linkNode.setAttribute(CONSTANTS.DWTempId, this.callback_counter);
        // attach a temp ID and store them so that once we get correct dwid's we replace the values.
        this.callback_map[this.callback_counter] = linkNode;
        this.callback_counter++;
        elem.setAttribute(CONSTANTS.DWTempId, this.callback_counter);
        this.callback_map[this.callback_counter] = elem;
        newCallbackObj.img_callback_id = this.callback_counter;
        this.callback_counter++;
        target = linkNode;
    } else {
        linkNode = elem.parentNode;
        imageId = linkNode.getAttribute(CONSTANTS.DWUniqueId);
        // there is no anchor tag associated with this img --> just return
        if (linkNode.tagName !== "A") {
            return;
        }

        // there is an anchor tag associated we need to delete it.
        var parentNode = linkNode.parentNode;
        var childNodesOfA = linkNode.childNodes,
            length = childNodesOfA.length,
            i = length - 1, // length will atleast be 1 as we know there is a child img
            prevElem = null;
        for (i; i >= 0; i--) {
            var curElem = childNodesOfA[i];
            if (i === length - 1) {
                parentNode.replaceChild(curElem, linkNode);
            } else {
                parentNode.insertBefore(curElem, prevElem);
            }
            if (curElem.nodeType === 1) {
                // this is an element
                curElem.setAttribute(CONSTANTS.DWTempId, this.callback_counter);
                newCallbackObj.img_callback_id = this.callback_counter;
                this.callback_map[this.callback_counter] = curElem;
                this.callback_counter++;
            }
            if (curElem.nodeType === 1) {
                newOuterHTML = curElem.outerHTML + newOuterHTML;
            } else if (curElem.nodeType === 3) {
                newOuterHTML = curElem.data + newOuterHTML;
            }
            prevElem = curElem;
        }
        target = elem;
    }

    // get details of parent and push it to SM layer.
    var parentId = target.parentNode.getAttribute(CONSTANTS.DWUniqueId);
    var newText = target.outerHTML;
    if (newOuterHTML !== "") {
        newText = newOuterHTML;
    }
    newCallbackObj.parentId = parentId;
    newCallbackObj.imgId = imageId;
    newCallbackObj.newText = newText;
    newCallbackObj.counter = this.callback_counter;
    newCallbackObj.callback = function (newObj) {
        this.updateDWIdsForAnchors(newObj);
    }.bind(this);

    if (this.fileSavePending >= 1) {
        this.fileSavePending++;
    }
    dwObject.DWSMGetDWIdForAnchorTag(newCallbackObj);
    dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.ELV_IH, DWLE_IMAGEHUD_HEADLIGHTS.LINK);
    this.updateTargetDropdown();
};

DwImageHud.prototype.shouldHide = function () {
    'use strict';
    if ((this.fileSavePending > 1)  || this.stopHide) {
        return false;
    }
    
    return true;
};

/**
 * @private updateDWIdsForAnchors
 * will be called from sm layer with newObj containing tempId's and their corresponding dwid's
 * @Param newObj: contains a map of tempId,dwid
 */
DwImageHud.prototype.updateDWIdsForAnchors = function (newObj) {
    'use strict';
//    if (!newObj) {
//    }
        
    var i, elem;
    for (i = 0; i < this.callback_counter; i++) {
        elem = this.callback_map[i];
        elem.setAttribute(CONSTANTS.DWUniqueId, newObj[i]);
        elem.removeAttribute(CONSTANTS.DWTempId);
    }
    // update target always after updating the link
    this.updateTarget();
    this.callback_counter = 0;
    
    if (this.fileSavePending > 1) {
        this.fileSavePending--;
        if (this.fileSavePending === 1) {
            window.parent.DWSaveDocument();
            this.fileSavePending--;
        }
    }
};
/**
 * @private updateAlt
 * update Alt attribute of the image
 * @Param none
 */
DwImageHud.prototype.updateAlt = function () {
    'use strict';
    this.updateElement("alt", DWLE_IMG_IDS.alt);
};
/**
 * @private updateTitle
 * update title attribute of the image
 * @Param none
 */
DwImageHud.prototype.updateTitle = function () {
    'use strict';
    this.updateElement("title", DWLE_IMG_IDS.title);
};
/**
 * @private updateWidth
 * update width attribute of the image
 * @Param none
 */
DwImageHud.prototype.updateWidth = function (updateUI) {
    'use strict';
    this.updateElement("width", DWLE_IMG_IDS.width);
    if (updateUI === true) {
        AuxiliaryExtensionObject.prototype.elementDimensionChanged.call(this);
//		if (this.attributeChanged) {
//			this.m_liveEditEvent.fire({
//				type: DW_LIVEEDIT_EVENTS.ElementChanged
//			});
//			this.attributeChanged = false;
//		}
        this.focusHudField(DWLE_IMG_IDS.width);
    }
};
/**
 * @private updateHeight
 * update height attribute of the image
 * @Param none
 */
DwImageHud.prototype.updateHeight = function (updateUI) {
    'use strict';
    this.updateElement("height", DWLE_IMG_IDS.height);
    if (updateUI === true) {
        AuxiliaryExtensionObject.prototype.elementDimensionChanged.call(this);
//		if (this.attributeChanged) {
//			this.m_liveEditEvent.fire({
//				type: DW_LIVEEDIT_EVENTS.ElementChanged
//			});
//			this.attributeChanged = false;
//		}
        this.focusHudField(DWLE_IMG_IDS.height);
	}
};
/**
 * @private updateTarget
 * update target attribute of the image
 * @Param none
 */
DwImageHud.prototype.updateTarget = function () {
    'use strict';

    var elem = this.getCurrentElement();
    if (!elem) {
        return;
    }
    var parent = elem.parentNode;
    if (parent.tagName === "A") {
        var value = document.getElementById(DWLE_IMG_IDS.target).value;
        if (value !== "none") {
            parent.setAttribute("target", value);
        } else {
            value = "";
            parent.removeAttribute("target");
        }

        var self = this;
        var funcCallback = function () {
            if (self.fileSavePending > 1) {
                self.fileSavePending--;
                if (self.fileSavePending === 1) {
                    window.parent.DWSaveDocument();
                    self.fileSavePending--;
                }
            }
        };
        if (this.fileSavePending >= 1) {
            this.fileSavePending++;
        }
        var imageTagId = elem.getAttribute(CONSTANTS.DWUniqueId);
        if (imageTagId) {
            dwObject.updateDWDocumentElementAttr(parent, 'target', value, funcCallback, imageTagId);
        }
        dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.ELV_IH, DWLE_IMAGEHUD_HEADLIGHTS.TARGET);
    } else {
        return;
    }
};
/**
 * @private updateElement
 * update given attribute of the image with the value in the input field with id attrId
 * @Param attr image attribute under edit
 * @Param attrId the id of the input field in the HUD whose value corresponds to given attribute
 */
DwImageHud.prototype.updateElement = function (attr, attrId) {
    'use strict';
    var elem = this.getCurrentElement();
    var targetElement = document.getElementById(attrId);
    var value = targetElement.value;
    targetElement.title = value;
    var curVal = elem.getAttribute(attr);
    if ((!value || typeof value === "undefined") && (!curVal || typeof curVal === "undefined")) {
        return;
    }
    if (value !== curVal) {
        if (value && value.length) {
            elem.setAttribute(attr, value);
        } else {
            if (attr === "src") {
                elem.setAttribute(attr, "temp value to clear cached drawing of image in CEF");
            }
            elem.removeAttribute(attr);
        }
        
        var self = this;
        var funcCallback = function () {
            if (self.fileSavePending > 1) {
                self.fileSavePending--;
                if (self.fileSavePending === 1) {
                    window.parent.DWSaveDocument();
                    self.fileSavePending--;
                }
            }
        };
        
        if (this.fileSavePending >= 1) {
            this.fileSavePending++;
        }
        dwObject.updateDWDocumentElementAttr(this.m_currentSelectedElement, attr, value, funcCallback);
        if (attr === "src") {
            dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.ELV_IH, DWLE_IMAGEHUD_HEADLIGHTS.SRC);
        } else if (attr === "title") {
            dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.ELV_IH, DWLE_IMAGEHUD_HEADLIGHTS.TITLE);
        } else if (attr === "alt") {
            dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.ELV_IH, DWLE_IMAGEHUD_HEADLIGHTS.ALT);
        } else if (attr === "height") {
            dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.ELV_IH, DWLE_IMAGEHUD_HEADLIGHTS.HEIGHT);
        } else if (attr === "width") {
            dwObject.logHeadlightsData(DWLE_IMAGEHUD_HEADLIGHTS.ELV_IH, DWLE_IMAGEHUD_HEADLIGHTS.WIDTH);
        }
		this.attributeChanged  = true;
    }
};

/**
 * @private validateInputText
 * input Text for width and height needs to be numeric.
 * @Param evt     : keystroke event that we received.
 */
DwImageHud.prototype.validateInputText = function (evt) {
    'use strict';
    var theEvent = evt || window.event;
    var key = theEvent.keyCode || theEvent.which;
    key = String.fromCharCode(key);
    var regex = /[0-9]|\./;
    if (!regex.test(key)) {
        theEvent.returnValue = false;
        if (theEvent.preventDefault) {
            theEvent.preventDefault();
        }
    }
};


var initDwImageHud = function () {
    'use strict';
    window.dwImageHudObj = new DwImageHud();
    window.dwImageHudObj.initialize();
};
